// Number of outer filter taps
#define NUM_DOF_TAPS 12

float4 viewportScale;
float4 viewportBias;

float maxCoC;

float2 filterTaps[NUM_DOF_TAPS];

texture tSceneColor;
texture tBlurDepth;

///////////////////////////////////////////////////////////////////////////////

struct VS_INPUT
{
	float4 vPos: POSITION;
	float2 vTexCoord: TEXCOORD;
};

struct VS_OUTPUT
{
	float4 vPos: POSITION;
	float2 vTexCoord: TEXCOORD;
};

///////////////////////////////////////////////////////////////////////////////

struct PS_INPUT
{
	float2 vTexCoord: TEXCOORD;
};

sampler SceneColorSampler = sampler_state
{
	texture = (tSceneColor);

//	MinFilter = Point;
//	MagFilter = Point;
	
	MinFilter = Linear;//Point;
	MagFilter = Linear;//Point;
	AddressU = Clamp;
	AddressV = Clamp;
};

sampler DepthBlurSampler = sampler_state
{
	texture = (tBlurDepth);
	MinFilter = Linear;//Point;
	MagFilter = Linear;//Point;
	
	//MinFilter = Point;
	//MagFilter = Point;
	AddressU = Clamp;
	AddressV = Clamp;
};

///////////////////////////////////////////////////////////////////////////////

// Screen space vertex shader
VS_OUTPUT filter_dof_vs(VS_INPUT v)
{
	VS_OUTPUT o;

	// Scale and bias viewport
	o.vPos = v.vPos * viewportScale + viewportBias;

	// Output tex. coordinates
	o.vTexCoord = v.vTexCoord;

	return o;
}

// DoF filter shader
float4 filter_dof_ps(PS_INPUT v) : COLOR
{
	// Get center sample
	float4 colorSum = tex2D(SceneColorSampler, v.vTexCoord);
	float2 centerDepthBlur = tex2D(DepthBlurSampler, v.vTexCoord);

	// Compute CoC size based on blurriness
	float sizeCoC = centerDepthBlur.y * maxCoC;

	float totalContribution = 1.0f;

	// Run through all taps
	for (int i = 0; i < NUM_DOF_TAPS; i++)
	{
		// Compute tap coordinates
		float2 tapCoord = v.vTexCoord + filterTaps[i] * sizeCoC;

		// Fetch tap sample
		float4 tapColor = tex2D(SceneColorSampler, tapCoord);
		float2 tapDepthBlur = tex2D(DepthBlurSampler, tapCoord);

		// Compute tap contribution
		float tapContribution = (tapDepthBlur.x > centerDepthBlur.x) ? 1.0f : tapDepthBlur.y;

		// Accumulate color and contribution
		colorSum += tapColor * tapContribution;
		totalContribution += tapContribution;
	}

	// Normalize to get proper luminance
	float4 finalColor = colorSum / totalContribution;

	return finalColor;
}

technique FilterDoF
{
	pass P0
	{
		VertexShader = compile vs_2_0 filter_dof_vs();
		PixelShader = compile ps_2_0 filter_dof_ps();

		CullMode = NONE;
	}
}

///////////////////////////////////////////////////////////////////////////////

float depthScale;

// Helper shader for visualizing depth/blurriness
float4 depth_blur_overlay_ps(PS_INPUT v) : COLOR
{
	float4 BlurDepth = tex2D(DepthBlurSampler, v.vTexCoord);

	return float4(BlurDepth.x * depthScale, BlurDepth.y, 0, 0);
}

technique DepthBlurOverlay
{
	pass P0
	{        
		VertexShader = compile vs_2_0 filter_dof_vs();
		PixelShader = compile ps_2_0 depth_blur_overlay_ps();

		CullMode = NONE;
	}
}